/*
 * Intel QV Linux kernel driver
 * Copyright (c) 1999 - 2012, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/*
 * Module Name:
 *   linuxdriveros_i.c
 *
 * Abstract:
 *   This is the driver OSI portion of NAL Linux.  These same
 *   function implementations will appear in Linuxlibrary.c. This file
 *   provides the necessary means to call the IOCTL needed and returns
 *   the correct values for the OS Interface of the NAL for Linux.
 *
 *
 * VSS Revision Control Information:
 * ---------------------------------
 *   $Workfile: linuxdriveros_i.c $
 *   $Date: 2012/01/16 13:19:47 $
 *   $Archive: /Quartzville2.0/nal/src/linux/driver/linuxdriveros_i.c $
 *   $Revision: 1.24 $
 */

#include "nalcodes.h"
#include "naltypes.h"
#include "os_i.h"
#include "pci_i.h"
#include "linuxnaldriver.h"

#include <asm/io.h>             /* for I/O routines, memory mapping. */
#include <asm/mman.h>           /* for memory access defines */
#include <linux/slab.h>         /* Allocation of memory */
#include <asm/atomic.h>         /* Atomic operations */
#include <linux/sched.h>        /* Interrupt routines */
#include <asm/page.h>           /* Page size definition */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/pci_ids.h>
#include <linux/errno.h>
#include <linux/fs.h>           /* Memory mapping functions */
#include <linux/mm.h>           /* Memory mapping functions */
#include <asm/uaccess.h>
#include <linux/pci.h>
#include <linux/delay.h>

spinlock_t   Global_AtomicTestSetSpinLock;

#if 0
BOOLEAN                  Global_DebugPrintEnabled = TRUE;
#else
BOOLEAN                  Global_DebugPrintEnabled = FALSE;
#endif

/***************************************************************************
**
** Name:            NalReadPort8()
**
** Description:     Reads an 8 bit value from a specified I/O Port
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Port = the I/O port from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT8
NalReadPort8(
    IN  PORT_ADDR Port
    )
{
    UINT8   Buffer = 0;
    Buffer = inb(Port);
    return Buffer;
}

/***************************************************************************
**
** Name:            NalReadPort16()
**
** Description:     Reads a 16bit value from a specified I/O Port
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Port = the I/O port from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT16
NalReadPort16(
    IN  PORT_ADDR Port
    )
{
    UINT16  Buffer = 0;

    Buffer = inw(Port);
    return Buffer;
}

/***************************************************************************
**
** Name:            NalReadPort32()
**
** Description:     Reads a 32bit value from a specified I/O Port
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Port = the I/O port from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT32
NalReadPort32(
    IN  PORT_ADDR Port
    )
{
    UINT32   Buffer = 0;
    Buffer = inl(Port);
    return Buffer;
}

/***************************************************************************
**
** Name:            NalWritePort8()
**
** Description:     Writes an 8bit value to a specified I/O Port
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Port  = the I/O port to which to write.
**                  Value = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWritePort8(
    IN  PORT_ADDR   Port,
    IN  UINT8       Value
    )
{
    outb(Value, Port);
    return TRUE;
}

/***************************************************************************
**
** Name:            NalWritePort16()
**
** Description:     Writes a 16bit value to a specified I/O Port
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Port  = the I/O port to which to write.
**                  Value = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWritePort16(
    IN  PORT_ADDR   Port,
    IN  UINT16      Value
    )
{
    outw(Value, Port);
    return TRUE;
}

/***************************************************************************
**
** Name:            NalWritePort32()
**
** Description:     Writes a 32 bit value to a specified I/O Port
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Port  = the I/O port to which to write.
**                  Value = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWritePort32(
    IN  PORT_ADDR   Port,
    IN  UINT32      Value
    )
{
    outl(Value, Port);
    return TRUE;
}

/***************************************************************************
**
** Name:            NalReadRegister8()
**
** Description:     Reads an 8 bit value from the specified memory address
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Address  = the memory address from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT8
NalReadRegister8(
    IN  KVOID*   Address
    )
{
    UINT8   Value = 0;

    if(Address != NULL)
    {
        Value = readb((CHAR*)Address);
    }
    return Value;
}

/***************************************************************************
**
** Name:            NalReadRegister16()
**
** Description:     Reads a 16 bit value from the specified memory address
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Address  = the memory address from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT16
NalReadRegister16(
    IN  KVOID*    Address
    )
{
    UINT16   Value = 0;

    if(Address != NULL)
    {
        Value = readw((CHAR*)Address);
    }
    return Value;
}

/***************************************************************************
**
** Name:            NalReadRegister32()
**
** Description:     Reads a 32bit value from the specified memory address
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Address  = the memory address from which to read.
**
** Returns:         The data read.
**
****************************************************************************/
UINT32
NalReadRegister32(
    IN  KVOID*        Address
    )
{
    UINT32   Value = 0;

    if(Address != NULL)
    {
        Value = readl((CHAR*)Address);
    }
    return Value;
}

/***************************************************************************
**
** Name:            NalWriteRegister8()
**
** Description:     Writes an 8bit value to a specified Memory Address.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Address  = Pointer to the address to write to.
**                  Value    = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWriteRegister8(
    IN  KVOID*                   Address,
    IN  UINT8                    Value
    )
{
    BOOLEAN Success = FALSE;

    if(Address != NULL)
    {
        writeb(Value, (CHAR*)Address);
        Success = TRUE;
    }
    return Success;
}

/***************************************************************************
**
** Name:            NalWriteRegister16()
**
** Description:     Writes a 16bit value to a specified Memory Address.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Address  = Pointer to the address to write to.
**                  Value    = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWriteRegister16(
    IN  KVOID*                Address,
    IN  UINT16                Value
    )
{
    BOOLEAN Success = FALSE;

    if(Address != NULL)
    {
        writew(Value, (CHAR*)Address);
        Success = TRUE;
    }
    return Success;
}

/***************************************************************************
**
** Name:            NalWriteRegister32()
**
** Description:     Writes a 32bit value to a specified Memory Address.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Address  = Pointer to the address to write to.
**                  Value    = the value to write.
**
** Returns:         TRUE if written, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
NalWriteRegister32(
    IN  KVOID*                   Address,
    IN  UINT32                   Value
    )
{
    BOOLEAN Success = FALSE;

    if(Address != NULL)
    {
        writel(Value, (CHAR*)Address);
        Success = TRUE;
    }
    return Success;
}

/***************************************************************************
**
** Name:            NalMmapAddress()
**
** Description:     Maps a physical address to a virtual address for I/O.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       VirtualAddress  = Pointer to the Virtual Address to map to
**                  PhysicalAddress = The address to map
**                  Length          = Number of bytes to map
**
** Returns:         NAL_STATUS
**                  Length contains the number of bytes actually mapped.
**                  VirtualAddress contains the base address of the mapped region
**
****************************************************************************/
NAL_STATUS
NalMmapAddress(
    IN OUT  KVOID**                 VirtualAddress,
    IN      NAL_PHYSICAL_ADDRESS    PhysicalAddress,
    IN OUT  UINT32*                 Length
    )
{
    NAL_STATUS NalStatus = NAL_INVALID_PARAMETER;
    if(VirtualAddress != NULL && Length != NULL)
    {
        *VirtualAddress = ioremap(PhysicalAddress, (UINT64)(*Length));
        if(*VirtualAddress != NULL)
        {
            NalStatus = NAL_SUCCESS;
        }
        else
        {
            NalStatus = NAL_MMAP_FAILED;
        }
    }
    return NalStatus;
}

/***************************************************************************
**
** Name:            NalUnmapAddress()
**
** Description:     Unmaps an address mapped with NalMmapAddress().
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       VirtualAddress  = Pointer to the Virtual Address to map to
**                  PhysicalAddress = The physical address mapped to the
**                                    VirtualAddress parameter.
**                  Length          = Number of bytes mapped.
**
** Returns:         NAL_STATUS Value
**
****************************************************************************/
NAL_STATUS
NalUnmapAddress(
    IN  KVOID*                  VirtualAddress,
    IN  NAL_PHYSICAL_ADDRESS    PhysicalAddress,
    IN  UINT32                  Length
    )
{
    NAL_STATUS NalStatus = NAL_INVALID_PARAMETER;
    if(VirtualAddress != NULL)
    {
        iounmap(VirtualAddress);
        NalStatus = NAL_SUCCESS;
    }
    return NalStatus;
}

/***************************************************************************
**
** Name:            NalDelayMilliseconds()
**
** Description:     Stops the thread for a specified amt of milliseconds
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Milliseconds = Number of milliseconds to pause.
**
** Returns:         Nothing.
**
****************************************************************************/
VOID
NalDelayMilliseconds(
    IN  UINT32 Milliseconds
    )
{
    mdelay(Milliseconds);
}

/***************************************************************************
**
** Name:            NalDelayMicroseconds()
**
** Description:     Stops the thread for a specified amt of microseconds
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Microseconds = Number of microseconds to pause.
**
** Returns:         Nothing.
**
****************************************************************************/
VOID
NalDelayMicroseconds(
    IN  UINT32 Microseconds
    )
{
    /* If more than 1000, use NalDelayMilliseconds */
    if(Microseconds>1000)
    {
        NalDelayMilliseconds((UINT32)(Microseconds/1000));
        Microseconds %= 1000;
    }
    udelay(Microseconds);
}

/***************************************************************************
**
** Name:            NalGetTimeStamp()
**
** Description:     Returns the TSC (Time Stamp Counter - IA32) or the ITC
**                  (Interval Time Counter - IA64).
**
** Author:          MJP
**
** Born on Date:    09/10/2001
**
** Arguments:       NONE
**
** Returns:         UINT64 - 64-bit time stamp that represents the number of
**                      clock increments since the CPU has started.
**
****************************************************************************/
UINT64
NalGetTimeStamp(    VOID    )
{
    UINT32  Low  = 0;
    UINT32  High = 0;

   // rdtsc(Low, High);
    return (UINT64)((((UINT64)High) << 32) | Low);
}

/***************************************************************************
**
** Name:            NalGetTimeStampsPerMicrosecond()
**
** Description:     Returns the number of time stamps in each microsecond.
**
** Author:          MJP
**
** Born on Date:    01/30/2002
**
** Arguments:       NONE
**
** Returns:         UINT64 - The number of time stamps that will occur during
**                           a microsecond.
**
** Note:            This is also implemented in user-mode of the Linux OS Interface.
**                  It is only included here because the NAL IOCTL interface requires it.
****************************************************************************/
UINT64
NalGetTimeStampsPerMicrosecond(    VOID    )
{
    return (UINT64)0;
}

/***************************************************************************
**
** Name:            _NalAllocateMemory()
**
** Description:     Allocates the specified number of pagable bytes.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       ByteCount  = Number of bytes to allocate.
**                  NamedLocator = Name such as File name, __FILE__ works.
**                  LineNumber = Line Number in the code file. __LINE__ works.
**
** Returns:         Pointer to the memory buffer, NULL if failed.
**
** Note:            This is used ONLY to allocate kernel memory from functions within
**                  the kernel portion of the NAL.  No IOCTL call will resolve to this.
**                  Instead, they will use malloc.
****************************************************************************/
VOID*
_NalAllocateMemory(
    IN  UINT32      ByteCount,
    IN  CHAR*       NamedLocator,
    IN  UINT32      LineNumber
    )
{
    VOID*   Address = kmalloc(ByteCount, GFP_KERNEL); //vmalloc(ByteCount);
    if(Address != NULL)
    {
        memset(Address, 0, ByteCount);
    }
    return Address;
}

/***************************************************************************
**
** Name:            _NalAllocateMemoryNonPaged()
**
** Description:     Allocates the specified number of bytes. These bytes are contiguous
**                  non-paged.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       ByteCount  = Number of bytes to allocate.
**                  Alignment = 32bit alignment requirements such as 4096.
**                  PhysicalAddress = Will contain the physical address of allocated
**                                    memory block. If this call fails, this will
**                                    be set to NULL. It is optional, so if it is
**                                    NULL, it will be ignored.
**                  NamedLocator = ASCII string that can identify filename or function
**                                 name where memory allocation is performed.
**                  LineNumber = The line number in NamedLocator where memory's allocated
**
** Returns:         Pointer to the memory buffer, NULL if failed.
**
****************************************************************************/
KVOID*
_NalAllocateMemoryNonPaged(
    IN  UINT32                  ByteCount,
    IN  UINT32                  Alignment,
    OUT NAL_PHYSICAL_ADDRESS*   PhysicalAddress, /* OPTIONAL */
    IN  CHAR*                   NamedLocator,
    IN  UINT32                  LineNumber
    )
{
    KVOID*  Allocation = NULL;

    /* Linux will automatically align the allocation to the nearest page size. If the
     * requested alignment is PAGE_SIZE or less, this works automagically and no alignment
     * is required. If not, we'll increase the alignment to the next page size multiple
     * so that it will automatically be aligned correctly. */
    NalDebugPrint("PAGE_SIZE = %d\n", PAGE_SIZE);
    if(ByteCount < Alignment)
    {
        NAL_LINUX_MEMORY_ROUNDUP(ByteCount, Alignment);
        NalDebugPrint("Aligned bytecount = %d\n", ByteCount);
    }

    /* Max allocation size allowed is 128KB by Linux in a single allocation. This is
     * because all usable non-paged allocation types are contiguous memory and Linux
     * limits this to a single 128kb max on 2.4 kernels. */
    if(ByteCount <= (128 * 1024))
    {
        Allocation = kmalloc(ByteCount, GFP_KERNEL | GFP_DMA);

        /* Get the physical address. */
        if(Allocation != NULL && PhysicalAddress != NULL)
        {
            *PhysicalAddress = (NAL_PHYSICAL_ADDRESS)__pa(Allocation);
        }
    }

    if(Allocation == NULL)
    {
        NalDebugPrint("Allocation failed: File: %s, Line %d\n", NamedLocator, LineNumber);
    }
    else
    {
        memset(Allocation, 0, ByteCount);
        if(PhysicalAddress != NULL)
        {
            NalDebugPrint("Phys Addr: %x", *PhysicalAddress);
        }
        NalDebugPrint("\n");
    }
    return Allocation;
}

/***************************************************************************
**
** Name:            _NalAllocateMemoryNonPagedPci()
**
** Description:     Allocates the specified number of bytes. These bytes are contiguous
**                  non-paged.
**
** Author:          AS
**
** Born on Date:    03/11/2009
**
** Arguments:       ByteCount  = Number of bytes to allocate.
**                  Alignment = 32bit alignment requirements such as 4096.
**                  PhysicalAddress = Will contain the physical address of allocated
**                                    memory block. If this call fails, this will
**                                    be set to NULL. It is optional, so if it is
**                                    NULL, it will be ignored.
**                  NamedLocator = ASCII string that can identify filename or function
**                                 name where memory allocation is performed.
**                  LineNumber = The line number in NamedLocator where memory's allocated
**
** Returns:         Pointer to the memory buffer, NULL if failed.
**
****************************************************************************/
KVOID*
_NalAllocateMemoryNonPagedPci(
    IN  KVOID*                  PDev,
    IN  UINT32                  ByteCount,
    IN  UINT32                  Alignment,
    OUT NAL_PHYSICAL_ADDRESS*   PhysicalAddress, /* OPTIONAL */
    IN  CHAR*                   NamedLocator,
    IN  UINT32                  LineNumber
    )
{
    KVOID*              Allocation      = NULL;
    struct pci_dev*     LinuxPciDevice  = (struct pci_dev*)PDev;
    dma_addr_t          Dma;

    /* Linux will automatically align the allocation to the nearest page size. If the
     * requested alignment is PAGE_SIZE or less, this works automagically and no alignment
     * is required. If not, we'll increase the alignment to the next page size multiple
     * so that it will automatically be aligned correctly. */
    NalDebugPrint("PAGE_SIZE = %d\n", PAGE_SIZE);
    if(ByteCount < Alignment)
    {
        NAL_LINUX_MEMORY_ROUNDUP(ByteCount, Alignment);
        NalDebugPrint("Aligned bytecount = %d\n", ByteCount);
    }

    /* Max allocation size allowed is 128KB by Linux in a single allocation. This is
     * because all usable non-paged allocation types are contiguous memory and Linux
     * limits this to a single 128kb max on 2.4 kernels. */
    if((LinuxPciDevice != NULL) && ByteCount <= NAL_LINUX_MAX_CONTIGUOUS_MEMORY_ALLOCATION)
    {
        /* Cant use kmalloc, since the __Pa doesnot give the correct physical address on ARm architecture
         *  Use either  pci_alloc_consistent for the desc ring or
         * dma_alloc_coherent http://fxr.watson.org/fxr/source/Documentation/pci.txt?v=linux-2.6 */

        Allocation = pci_alloc_consistent(LinuxPciDevice,ByteCount, &Dma);
        *PhysicalAddress = (NAL_PHYSICAL_ADDRESS)Dma;
    }

    if(Allocation == NULL)
    {
        NalDebugPrint("Allocation failed: File: %s, Line %d\n", NamedLocator, LineNumber);
    }
    else
    {
        memset(Allocation, 0, ByteCount);
        if(PhysicalAddress != NULL)
        {
            NalDebugPrint("Phys Addr: %x", *PhysicalAddress);
        }
        NalDebugPrint("\n");
    }
    return Allocation;
}

/***************************************************************************
**
** Name:            _NalFreeMemory()
**
** Description:     Frees the memory allocated at the buffer address.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Address  = Pointer to the address to free.
**                  NamedLocator = Name such as File name, __FILE__ works.
**                  LineNumber = Line Number in the code file. __LINE__ works.
**
** Returns:         Nothing.
**
** Note:            This is used ONLY to free kernel memory allocated from within
**                  the kernel portion of the NAL.  No IOCTL call will resolve to this.
****************************************************************************/
VOID
_NalFreeMemory(
    IN  VOID*       Address,
    IN  CHAR*       NamedLocator,
    IN  UINT32      LineNumber
    )
{
    if(Address != NULL)
    {
        kfree(Address);
    }
}

/***************************************************************************
**
** Name:            _NalFreeMemoryNonPaged()
**
** Description:     Frees the memory allocated at the buffer address.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Address  = Pointer to the address to free.
**
** Returns:         Nothing.
**
****************************************************************************/
VOID
_NalFreeMemoryNonPaged(
    IN  KVOID*      Address,
    IN  CHAR*       NamedLocator,
    IN  UINT32      LineNumber
    )
{
    if(Address != NULL)
    {
        //free_page(Address);
        kfree(Address);
    }
}

/***************************************************************************
**
** Name:            NalFreeMemoryNonPagedPci()
**
** Description:     Frees the memory allocated at the buffer address.
**
** Author:          AS
**
** Born on Date:    03/20/2009
**
** Arguments:       Address  = Pointer to the address to free.
**
** Returns:         Nothing.
**
****************************************************************************/
VOID
NalFreeMemoryNonPagedPci(
    IN  KVOID*                  PDev,
    IN  KVOID*                  Address,
    IN  NAL_PHYSICAL_ADDRESS    PhysicalAddress,
    IN UINT32                   Size
    )
{
    struct pci_dev*     LinuxPciDevice  = (struct pci_dev*)PDev;

    if(PDev != NULL && Address != NULL && PhysicalAddress != 0)
    {
        printk(KERN_DEBUG "freeing memory with size %d and phy address %x'%x\n", Size, HIDWORD((UINT64)PhysicalAddress), LODWORD((UINT64)PhysicalAddress));
        pci_free_consistent(LinuxPciDevice, Size, Address, PhysicalAddress);
    }
}

/***************************************************************************
**
** Name:            NalGetPhysicalMemoryAddress()
**
** Description:     Returns the Physical address associated with this virtual
**                  address. This address can be used for DMA by HW.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Address  = Pointer to the address to resolve.
**
** Returns:         Physical Address.
**
** Note:            EFI maps physical addresses to their virtual addresses,
**                  so this returns the virtual address casted.
**
****************************************************************************/
NAL_PHYSICAL_ADDRESS
NalGetPhysicalMemoryAddress(
    IN  KVOID* VirtualAddress
    )
{
    NAL_PHYSICAL_ADDRESS NalPhysical = 0;

    if(VirtualAddress != NULL)
    {
        NalPhysical = __pa(VirtualAddress);
    }

    return NalPhysical;
}

/***************************************************************************
**
** Name:            NalKMemSet()
**
** Description:     Sets Dest to the value of Value for Size length.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Dest     = Pointer to the address to set
**                  Value    = Integer value to set.
**                  Size     = Bytes to set
**
** Returns:         Dest
**
****************************************************************************/
KVOID*
NalKMemset(
    IN      KVOID*      Dest,
    IN      int         Value,
    IN      UINTN       Size
    )
{
    return memset(Dest, Value, Size);
}

/***************************************************************************
**
** Name:            NalKtoUMemcpy()
**
** Description:     Copies a buffer from Kernel to User space of Size bytes.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Source     = Pointer to the Kernel Address.
**                  Dest       = Pointer to the User Address.
**                  Size       = Bytes to Copy
**
** Returns:         Dest
**
****************************************************************************/
VOID*
NalKtoUMemcpy(
    IN      VOID*       Dest,
    IN      KVOID*      Source,
    IN      UINTN       Size
    )
{
    UINTN   BytesLeft   = 0;
    BytesLeft = copy_to_user(Dest, Source, Size);
    return Dest;
}

/***************************************************************************
**
** Name:            NalKtoKMemcpy()
**
** Description:     Does a kernel mode memcpy between two kernel mode buffers.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Source     = Pointer to the Kernel Address.
**                  Dest       = Pointer to the Kenerl Address.
**                  Size       = Bytes to Copy
**
** Returns:         Dest
**
****************************************************************************/
KVOID*
NalKtoKMemcpy(
    IN      KVOID*      Dest,
    IN      KVOID*      Source,
    IN      UINTN       Size
    )
{
    return memcpy(Dest, Source, Size);
}

/***************************************************************************
**
** Name:            NalUtoKMemcpy()
**
** Description:     Copies a buffer from User to Kernel space of Size bytes.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Source     = Pointer to the User Address.
**                  Dest       = Pointer to the Kernel Address.
**                  Size       = Bytes to Copy
**
** Returns:         Dest
**
****************************************************************************/
KVOID*
NalUtoKMemcpy(
    IN      KVOID*      Dest,
    IN      VOID*       Source,
    IN      UINTN       Size
    )
{
    UINTN   BytesLeft   = 0;
    BytesLeft = copy_from_user(Dest, Source, Size);
    return Dest;
}

/***************************************************************************
**
** Name:            NalAtomicIncrement32()
**
** Description:     Atomically increments a UINT32 Pointer.
**
** Author:          MVM
**
** Born on Date:    03/31/2005
**
** Arguments:       Address  = Pointer to the address to increment
**
** Returns:         Value at incremented Address
**
** Note:            This assumes that the Address value is a user-mode value. This function cannot
**                  be called from within this driver to work on a kernel address. It's using
**                  copy_to_user/copy_from_user
****************************************************************************/
UINT32
NalAtomicIncrement32(
    IN  UINT32* Address
    )
{
    UINT32   ReturnValue  = 0;
    UINT32   AddressLocal = 0;
    UINTN    BytesLeft    = 0;

    if(Address != NULL)
    {
        /* Linux doesnt provide this API, so we use a SpinLock */
        spin_lock(&Global_AtomicTestSetSpinLock);
        BytesLeft = copy_from_user((void*)(&AddressLocal), (void*)Address, sizeof(UINT32));

        /* Increment the address */
        AddressLocal++;

        /* Return *Address regardless if changed or not. */
        ReturnValue = AddressLocal;

        BytesLeft = copy_to_user((void*)Address, (void*)(&AddressLocal), sizeof(UINT32));

        /* Release the spinlock */
        spin_unlock(&Global_AtomicTestSetSpinLock);
    }
    return ReturnValue;
}

/***************************************************************************
**
** Name:            NalAtomicDecrement32()
**
** Description:     Atomically decrements a UINT32 Pointer.
**
** Author:          MVM
**
** Born on Date:    03/31/2005
**
** Arguments:       Address  = Pointer to the address to decrement
**
** Returns:         Value at decremented Address
**
** Note:            This assumes that the Address value is a user-mode value. This function cannot
**                  be called from within this driver to work on a kernel address. It's using
**                  copy_to_user/copy_from_user
****************************************************************************/
UINT32
NalAtomicDecrement32(
    IN  UINT32* Address
    )
{
    UINT32  ReturnValue  = 0;
    UINT32  AddressLocal = 0;
    UINTN   BytesLeft    = 0;

    if(Address != NULL)
    {
        /* Linux doesnt provide this API, so we use a SpinLock */
        spin_lock(&Global_AtomicTestSetSpinLock);
        BytesLeft = copy_from_user((void*)(&AddressLocal), (void*)Address, sizeof(UINT32));

        /* Decrement the address */
        AddressLocal--;

        /* Return *Address regardless if changed or not. */
        ReturnValue = AddressLocal;

        BytesLeft = copy_to_user((void*)Address, (void*)(&AddressLocal), sizeof(UINT32));

        /* Release the spinlock */
        spin_unlock(&Global_AtomicTestSetSpinLock);
    }

    return ReturnValue;
}

/***************************************************************************
**
** Name:            NalAtomicTestSet32()
**
** Description:     If "Address" contains the same value as "Test", this will
**                  set "Address" to contain "Set" instead.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       Address  = Pointer to the address to test & set
**                  Test     = 32bit value to test
**                  Set      = 32bit value to set
**
** Returns:         Value at prior to being modified Address
**
** Note:            This assumes that the Address value is a user-mode value. This function cannot
**                  be called from within this driver to work on a kernel address. It's using
**                  copy_to_user/copy_from_user
**
****************************************************************************/
UINT32
NalAtomicTestSet32(
    IN  UINT32* Address,
    IN  UINT32  Test,
    IN  UINT32  Set
    )
{
    UINT32  ReturnValue  = 0;
    UINT32  AddressLocal = 0;
    UINTN   BytesLeft    = 0;

    if(Address != NULL)
    {
        /* Linux doesnt provide this API, so we use a SpinLock */
        spin_lock(&Global_AtomicTestSetSpinLock);
        BytesLeft = copy_from_user((void*)(&AddressLocal), (void*)Address, sizeof(UINT32));

        /* Return *Address regardless if changed or not. */
        ReturnValue = AddressLocal;

        /* Test now, then set if true. */
        if(AddressLocal == Test)
        {
            AddressLocal = Set;
        }

        BytesLeft = copy_to_user((void*)Address, (void*)(&AddressLocal), sizeof(UINT32));

        /* Release the spinlock */
        spin_unlock(&Global_AtomicTestSetSpinLock);
    }


    return ReturnValue;
}

/***************************************************************************
**
** Name:            NalDebugPrintSupport()
**
** Description:     Returns debugging capabilities
**
** Author:          MVM
**
** Born on Date:    04/10/2001
**
** Arguments:       DebugCapabilities = Will receive the capabilities
**
** Returns:         a NAL_STATUS Value
**
****************************************************************************/
NAL_STATUS
NalDebugPrintSupport(
    OUT UINT32* DebugCapabilities
    )
{
    if (DebugCapabilities != NULL)
    {
        *DebugCapabilities = NAL_DEBUG_PRINT_SUPPORTED |
                             NAL_DEBUG_PRINT_FILEIO;
    }

    return NAL_SUCCESS;
}

/***************************************************************************
**
** Name:            NalDebugPrint()
**
** Description:     Sends a text string out to the debugger.
**
** Author:          MVM
**
** Born on Date:    04/10/2001
**
** Arguments:       same as printf()
**
** Returns:         NAL_STATUS Value
**
****************************************************************************/
NAL_STATUS
NalDebugPrint(
    IN CHAR* Format,
    ...
    )
{
    CHAR        FormattedStatus[500];
    va_list     ArgPtr;
    NAL_STATUS  DbgPrintStatus = NAL_DEBUGPRINT_FAILED;

    if(Global_DebugPrintEnabled == TRUE)
    {
        /* Generate the formatted debug message. */
        va_start(ArgPtr, Format);
        vsprintf(FormattedStatus, Format, ArgPtr);
        va_end(ArgPtr);

        /* Print the message to the file and set the correct status. */
        printk(KERN_DEBUG "%s", FormattedStatus);
        DbgPrintStatus = NAL_SUCCESS;
    }
    return DbgPrintStatus;
}

/***************************************************************************
**
** Name:            NalEnableDebugPrint()
**
** Description:     Sets Global_DebugPrintEnabled flag to whatever the
**                  value of "Enable" is currently set to.
**
** Author:          SPD
**
** Born on Date:    09/03/2002
**
** Arguments:       Enable      - Boolean flag that when set to TRUE, enables
**                                debug print support.
**
** Returns:         Nothing
**
****************************************************************************/
VOID
NalEnableDebugPrint(
    IN BOOLEAN Enable
    )
{
    Global_DebugPrintEnabled = Enable;
}

/* These functions aren't needed by the linux driver but are included because the IOCTL interface
 * is shared */
NAL_STATUS
NalMmapAddressEx(
    IN OUT  KVOID**                 VirtualAddress,
    IN      NAL_PHYSICAL_ADDRESS    PhysicalAddress,
    IN OUT  UINT32*                 Length,
    IN      UINTN                   ProcessId
    )
{
    return NalMmapAddress(VirtualAddress, PhysicalAddress, Length);
}

NAL_STATUS
NalUnmapAddressEx(
    IN  KVOID*                  VirtualAddress,
    IN  NAL_PHYSICAL_ADDRESS    PhysicalAddress,
    IN  UINT32                  Length,
    IN  UINTN                   ProcessId
    )
{
    return NalUnmapAddress(VirtualAddress, PhysicalAddress, Length);
}

KVOID*
_NalAllocateMemoryNonPagedEx(
    IN  UINT32                  ByteCount,
    IN  UINT32                  Alignment,
    IN  UINTN                   ProcessId,
    OUT NAL_PHYSICAL_ADDRESS*   PhysicalAddress, /* OPTIONAL */
    IN  CHAR*                   NamedLocator,
    IN  UINT32                  LineNumber
    )
{
    return _NalAllocateMemoryNonPaged(ByteCount, Alignment, PhysicalAddress, NamedLocator, LineNumber);
}

VOID
_NalFreeMemoryNonPagedEx(
    IN  KVOID*      Address,
    IN  UINTN       ProcessId,
    IN  CHAR*       NamedLocator,
    IN  UINT32      LineNumber
    )
{
    return _NalFreeMemoryNonPaged(Address, NamedLocator, LineNumber);
}

NAL_PHYSICAL_ADDRESS
NalGetPhysicalMemoryAddressEx(
    IN  KVOID*  VirtualAddress,
    IN  UINTN   ProcessId
    )
{
    return NalGetPhysicalMemoryAddress(VirtualAddress);
}

